package com.androidbook.pettracker3;

import android.content.ContentUris;
import android.content.ContentValues;
import android.content.Context;
import android.content.Intent;
import android.content.res.TypedArray;
import android.database.Cursor;
import android.database.sqlite.SQLiteQueryBuilder;
import android.net.Uri;
import android.os.Bundle;
import android.provider.MediaStore.Images.Media;
import android.view.View;
import android.view.ViewGroup;
import android.widget.AutoCompleteTextView;
import android.widget.BaseAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.FilterQueryProvider;
import android.widget.Gallery;
import android.widget.ImageView;
import android.widget.SimpleCursorAdapter;

import com.androidbook.pettracker3.PetTrackerDatabase.PetType;
import com.androidbook.pettracker3.PetTrackerDatabase.Pets;

public class PetTrackerEntryActivity extends PetTrackerActivity {
	protected Cursor mCursorAutoComplete;
	protected Cursor mCursorImages;

	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.petentry);

		// Wypełnienie kontrolki Gallery obrazkami z karty CD.
		fillGallery();

		// Wypełnienie listy automatycznie uzupełnianych słów na podsatwie zawartości z bazy
		fillAutoComplete();
		
		// Obsługa przycisku Zapisz
		final Button savePet = (Button) findViewById(R.id.ButtonSave);
		savePet.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {

				final EditText petName = (EditText) findViewById(R.id.EditTextName);
				final EditText petType = (EditText) findViewById(R.id.EditTextSpecies);
				long imageId = PetRecord.INVALID_PET_IMAGE_ID;
				String strImageUriPathString = "";

				final Gallery gall = (Gallery) findViewById(R.id.GalleryOfPics);
				ImageView selectedImageView = (ImageView) gall
						.getSelectedView();
				GalleryRecord galleryItem;
				if (selectedImageView != null) {
					galleryItem = (GalleryRecord)selectedImageView.getTag();
					imageId = galleryItem.getImageId();
					strImageUriPathString = galleryItem.getImageUriPath();
				}
				
				String strPetType = petType.getText().toString().toLowerCase();
				String strPetName = petName.getText().toString();
				PetRecord newRecord = new PetRecord(strPetName, strPetType, strImageUriPathString, imageId,
						PetRecord.INVALID_PET_ID);
				addPetRecord(newRecord);

				// Przywrócenie stanu początkowego formularza.
				petName.setText(null);
				petType.setText(null);
			}
		});

		// Obsługa przycisku Pokaż wszystkie zwierzaki
		final Button gotoList = (Button) findViewById(R.id.ButtonShowPets);
		gotoList.setOnClickListener(new View.OnClickListener() {
			public void onClick(View v) {

				// Przejście do aktywności wyświetlającej listę zwierzaków.
				Intent intent = new Intent(PetTrackerEntryActivity.this,
						PetTrackerListActivity.class);
				startActivity(intent);
			}
		});
	}

	// Wypełnienie kontrolki Gallery informacjami zwróconymi przez dostawcę treści -
	// a konkretnie wszystkimi obrazkami przechowywanymi na karcie SD. Kontrolka jest 
	// wypełniana tylko raz - w momencie uruchamiania aktywności. Informacje nie są 
	// odświeżane, choć dysponujemy kursorem (obiekten Cursor), więc z technicznego 
	// punktu widzenia, nic nie stoi na przeszkodzie, by ponownie pobrać dane.
	private void fillGallery() {
		
	    // Tablica określająca jakie kolumny należy zwracać.
		String[] projection = new String[] { Media._ID, Media.TITLE }; // choć tytułu nie używamy...

		// Bazowy URI treści na karcie SD.
		Uri mMedia = Media.EXTERNAL_CONTENT_URI;

		// Pobranie plików multimedialnych dostępnych na karcie SC (zapytanie wykonywane przez dostawcę treści)
		// Najlepszy sposób pobrania wyników zapytania - skorzystanie z zarządzanego kursora.
		mCursorImages = managedQuery(mMedia, projection, 				
				null, null, 
				Media.DATE_TAKEN + " ASC"); // Klauzula ORDER BY.

		// Wypełnienie naszego niestandardowego obiektu ImageUriAdapter na podstawie zawartości kursora. 
		ImageUriAdapter iAdapter = new ImageUriAdapter(this, mCursorImages, mMedia); // Interesują nas obrazki tylko na karcie SD.
		
		// Przypisanie adaptera do kontrolki, w celu wyświetlenia obrazków.
		final Gallery pictureGal = (Gallery) findViewById(R.id.GalleryOfPics);
		pictureGal.setAdapter(iAdapter);
	}

	
	private void fillAutoComplete() {
		mCursorAutoComplete = mDB.query(PetType.PETTYPE_TABLE_NAME,
				new String[] { PetType.PET_TYPE_NAME, PetType._ID }, null,
				null, null, null, PetType.DEFAULT_SORT_ORDER);

		startManagingCursor(mCursorAutoComplete);

		SimpleCursorAdapter adapter = new SimpleCursorAdapter(this, 
				android.R.layout.simple_dropdown_item_1line,
				mCursorAutoComplete,
				new String[]{PetType.PET_TYPE_NAME},
				new int[]{android.R.id.text1}			
		);
		
		// Chcemy umieścić tekst w kontrolka EditText, a nie TextView; dlatego 
		// implementujemy obiekty CursorToStringConverter oraz FilterQueryProvider.
		// CursorToStringConverter - kontroluje jaka "wartość" zostanie zwrócona po wybraniu opcji TextEdit.
		// W tym przypadku będzie to sam tekst, a nie jedgo identyfikator.
		// FilterQueryProvider - pozwala na zarządzanie procesem filtrowania, którym w naszym przypadku będzie niestandardowa funkcja dopasowująca.	
		adapter.setCursorToStringConverter(new MyCursorToStringConverter());
		adapter.setFilterQueryProvider(new MyFilterQueryProvider());
		
		AutoCompleteTextView text = (AutoCompleteTextView) findViewById(R.id.EditTextSpecies);
		text.setAdapter(adapter);


	}
	
	// Ta metoda kontroluje jaką kolumnę należy umieścić w polu edittext. Domyślne textview.tostring, nie jest zbyt pomocne.
	class MyCursorToStringConverter implements SimpleCursorAdapter.CursorToStringConverter {

		public CharSequence convertToString(Cursor cursor) {
			return cursor.getString(cursor.getColumnIndex(PetType.PET_TYPE_NAME));
		}
	}
	
	// Nasz niestandardowy filtr odnajduje wszysstkie pasujące do wzorca łańcuchy znaków, a nie tylko te, w których
	// dopasowanie jest umieszczone na początku łańcucha.
	// Zastosowano w niej niewielką sztuczkę, gdyż nie można prawidłowo odświeżać zawartości kursora.
	class MyFilterQueryProvider implements FilterQueryProvider {

		public Cursor runQuery(CharSequence constraint) {
			
			if ((constraint != null) && (mCursorAutoComplete != null))
			{		
				String strWhere =  PetType.PET_TYPE_NAME + " LIKE ?";	

				stopManagingCursor(mCursorAutoComplete);
				mCursorAutoComplete = mDB.query(PetType.PETTYPE_TABLE_NAME,
						new String[] { PetType.PET_TYPE_NAME, PetType._ID }, strWhere, new String[] { "%" + constraint.toString() + "%"},
						null, null, PetType.DEFAULT_SORT_ORDER);
				startManagingCursor(mCursorAutoComplete);
			}

			return mCursorAutoComplete;
		}	
	}
	
	// Dodanie odpowiednich rekordów do bazy danych (zwierzaków i gatunków).
	private void addPetRecord(PetRecord newRecord) {

	    // Zapis nowych rekordów. Ponieważ zapisywanych jest kilka rekordów, zatem operację realizujemy w ramach transakcji.
	    // Zatem zostaną zapisane wszystkie rekordy lub żaden z nich.
		mDB.beginTransaction();
		try {

			// Sprawdzenie czy gatunek istnieje.
			long rowPetTypeId = 0;

			// Zapytanie SQL -> "select * from table_pettype where PetType.pettype_name='string' 
			SQLiteQueryBuilder queryBuilder = new SQLiteQueryBuilder();
			queryBuilder.setTables(PetType.PETTYPE_TABLE_NAME);
			queryBuilder.appendWhere(PetType.PET_TYPE_NAME + "='" + newRecord.getPetType()
					+ "'");

			// Wyknanie zapytania, bo już jest gotowe.
			Cursor c = queryBuilder.query(mDB, null, null, null, null, null,
					null);

			if (c.getCount() == 0) {
				// Dodanie nowego gatunku do listy.
				ContentValues typeRecordToAdd = new ContentValues();
				typeRecordToAdd.put(PetType.PET_TYPE_NAME, newRecord.getPetType());
				rowPetTypeId = mDB.insert(PetType.PETTYPE_TABLE_NAME,
						PetType.PET_TYPE_NAME, typeRecordToAdd);

			} else {
			    // Gatunek już istnieje, pobieramy jego identyfikator - będzie nam później potrzebny.
				c.moveToFirst();
				rowPetTypeId = c.getLong(c.getColumnIndex(PetType._ID));
			}

			c.close();

			// Zawsze dodajemy nowy rekord, nawet jeśli zwierzak o tym samym imieniu już istnieje.
			ContentValues petRecordToAdd = new ContentValues();
			petRecordToAdd.put(Pets.PET_NAME, newRecord.getPetName());
			petRecordToAdd.put(Pets.PET_TYPE_ID, rowPetTypeId);
			petRecordToAdd.put(Pets.PET_IMAGE_URI, newRecord.getPetImageUriPath());
			petRecordToAdd.put(Pets.PET_IMAGE_ID, newRecord.getPetImageId());
			mDB.insert(Pets.PETS_TABLE_NAME, Pets.PET_NAME, petRecordToAdd);

			mDB.setTransactionSuccessful();
		} finally {
			mDB.endTransaction();
		}
	}

	// Nasz adapter przechowuje informacje o rekoradach obrazków wyświetlonych w kontrolce Gallery, czyli 
	// obrazków z karty SD.
	public class ImageUriAdapter extends BaseAdapter {
		int mGalleryItemBackground;
		private GalleryRecord[] mPics;
		private Context mContext;

		public ImageUriAdapter(Context c, Cursor curs, Uri baseUri) {
			mContext = c;

			int iNumPics = curs.getCount();
			mPics = new GalleryRecord[iNumPics];

			curs.moveToFirst();
			for (int i = 0; i < iNumPics; i++) {				
				final String strUri = baseUri.toString();
				final long iId = curs.getLong(curs.getColumnIndex(Media._ID));	
				mPics[i] = new GalleryRecord(strUri, iId);
				curs.moveToNext();
			}
			
			TypedArray a = obtainStyledAttributes(R.styleable.default_gallery);
			mGalleryItemBackground = a.getResourceId(
					R.styleable.default_gallery_android_galleryItemBackground, 0);
			a.recycle();
		}

		public int getCount() {
			return mPics.length;
		}

		public Object getItem(int position) {
			return mPics[position].getImageId(); 
		}

		public long getItemId(int position) {
			return mPics[position].getImageId();
		}

		// Zwraca obiekt View odpowiadający każdemu elementowi wyświetlanemu w kontrolce Gallery. 
		// Każdy taki element to po prostu obiekt ImageView.
		// Wskazówka: Używamy właściwości Tag kontrolki ImageView zapisując w nich przydatne metadane.
		public View getView(int position, View convertView, ViewGroup parent) {
			ImageView i = new ImageView(mContext);

		    Uri baseUri = Uri.parse(mPics[position].getImageUriPath());
			Uri myImageUri = ContentUris.withAppendedId(
					baseUri, mPics[position].getImageId());
			i.setImageURI(myImageUri);
			
			// Bardzo użyteczna sztuczka - zapisujemy metadane z których będziemy później korzystać.
			i.setTag(mPics[position]);
			
			// Wszystkie obrazki w galerii będą mieć takie same wymiary/proporcje.
			i.setMaxHeight(100);
			i.setMaxWidth(100);
			i.setAdjustViewBounds(true);

			// Preferowane tło elementów galerii.
			i.setBackgroundResource(mGalleryItemBackground);

			return i;
		}
	}
}